/*
* FileSaver.js
* A saveAs() FileSaver implementation.
*
* By Eli Grey, http://eligrey.com
*
* License : https://github.com/eligrey/FileSaver.js/blob/master/LICENSE.md (MIT)
* source  : http://purl.eligrey.com/github/FileSaver.js
*/

// The one and only way of getting global scope in all environments
// https://stackoverflow.com/q/3277182/1008999
var _global = typeof window === 'object' && window.window === window
  ? window : typeof self === 'object' && self.self === self
  ? self : typeof global === 'object' && global.global === global
  ? global
  : this

function bom (blob, opts) {
  if (typeof opts === 'undefined') opts = { autoBom: false }
  else if (typeof opts !== 'object') {
    console.warn('Deprecated: Expected third argument to be a object')
    opts = { autoBom: !opts }
  }

  // prepend BOM for UTF-8 XML and text/* types (including HTML)
  // note: your browser will automatically convert UTF-16 U+FEFF to EF BB BF
  if (opts.autoBom && /^\s*(?:text\/\S*|application\/xml|\S*\/\S*\+xml)\s*;.*charset\s*=\s*utf-8/i.test(blob.type)) {
    return new Blob([String.fromCharCode(0xFEFF), blob], { type: blob.type })
  }
  return blob
}

function download (url, name, opts) {
  var xhr = new XMLHttpRequest()
  xhr.open('GET', url)
  xhr.responseType = 'blob'
  xhr.onload = function () {
    saveAs(xhr.response, name, opts)
  }
  xhr.onerror = function () {
    console.error('could not download file')
  }
  xhr.send()
}

function corsEnabled (url) {
  var xhr = new XMLHttpRequest()
  // use sync to avoid popup blocker
  xhr.open('HEAD', url, false)
  try {
    xhr.send()
  } catch (e) {}
  return xhr.status >= 200 && xhr.status <= 299
}

// `a.click()` doesn't work for all browsers (#465)
function click (node) {
  try {
    node.dispatchEvent(new MouseEvent('click'))
  } catch (e) {
    var evt = document.createEvent('MouseEvents')
    evt.initMouseEvent('click', true, true, window, 0, 0, 0, 80,
                          20, false, false, false, false, 0, null)
    node.dispatchEvent(evt)
  }
}

// Detect WebView inside a native macOS app by ruling out all browsers
// We just need to check for 'Safari' because all other browsers (besides Firefox) include that too
// https://www.whatismybrowser.com/guides/the-latest-user-agent/macos
var isMacOSWebView = _global.navigator && /Macintosh/.test(navigator.userAgent) && /AppleWebKit/.test(navigator.userAgent) && !/Safari/.test(navigator.userAgent)

var saveAs = _global.saveAs || (
  // probably in some web worker
  (typeof window !== 'object' || window !== _global)
    ? function saveAs () { /* noop */ }

  // Use download attribute first if possible (#193 Lumia mobile) unless this is a macOS WebView
  : ('download' in HTMLAnchorElement.prototype && !isMacOSWebView)
  ? function saveAs (blob, name, opts) {
    var URL = _global.URL || _global.webkitURL
    var a = document.createElement('a')
    name = name || blob.name || 'download'

    a.download = name
    a.rel = 'noopener' // tabnabbing

    // TODO: detect chrome extensions & packaged apps
    // a.target = '_blank'

    if (typeof blob === 'string') {
      // Support regular links
      a.href = blob
      if (a.origin !== location.origin) {
        corsEnabled(a.href)
          ? download(blob, name, opts)
          : click(a, a.target = '_blank')
      } else {
        click(a)
      }
    } else {
      // Support blobs
      a.href = URL.createObjectURL(blob)
      setTimeout(function () { URL.revokeObjectURL(a.href) }, 4E4) // 40s
      setTimeout(function () { click(a) }, 0)
    }
  }

  // Use msSaveOrOpenBlob as a second approach
  : 'msSaveOrOpenBlob' in navigator
  ? function saveAs (blob, name, opts) {
    name = name || blob.name || 'download'

    if (typeof blob === 'string') {
      if (corsEnabled(blob)) {
        download(blob, name, opts)
      } else {
        var a = document.createElement('a')
        a.href = blob
        a.target = '_blank'
        setTimeout(function () { click(a) })
      }
    } else {
      navigator.msSaveOrOpenBlob(bom(blob, opts), name)
    }
  }

  // Fallback to using FileReader and a popup
  : function saveAs (blob, name, opts, popup) {
    // Open a popup immediately do go around popup blocker
    // Mostly only available on user interaction and the fileReader is async so...
    popup = popup || open('', '_blank')
    if (popup) {
      popup.document.title =
      popup.document.body.innerText = 'downloading...'
    }

    if (typeof blob === 'string') return download(blob, name, opts)

    var force = blob.type === 'application/octet-stream'
    var isSafari = /constructor/i.test(_global.HTMLElement) || _global.safari
    var isChromeIOS = /CriOS\/[\d]+/.test(navigator.userAgent)

    if ((isChromeIOS || (force && isSafari) || isMacOSWebView) && typeof FileReader !== 'undefined') {
      // Safari doesn't allow downloading of blob URLs
      var reader = new FileReader()
      reader.onloadend = function () {
        var url = reader.result
        url = isChromeIOS ? url : url.replace(/^data:[^;]*;/, 'data:attachment/file;')
        if (popup) popup.location.href = url
        else location = url
        popup = null // reverse-tabnabbing #460
      }
      reader.readAsDataURL(blob)
    } else {
      var URL = _global.URL || _global.webkitURL
      var url = URL.createObjectURL(blob)
      if (popup) popup.location = url
      else location.href = url
      popup = null // reverse-tabnabbing #460
      setTimeout(function () { URL.revokeObjectURL(url) }, 4E4) // 40s
    }
  }
)

_global.saveAs = saveAs.saveAs = saveAs

if (typeof module !== 'undefined') {
  module.exports = saveAs;
}


var batchDwonloadImagesLib = {
    // 传入图片路径，返回base64
    getBase64: function (img, filename) {
        var image = new Image();
        image.crossOrigin = 'Anonymous';
        image.src = img;
        var deferred = $.Deferred();
        if (img) {
            image.onload = function () {
                var canvas = document.createElement("canvas");
                canvas.width = image.width;
                canvas.height = image.height;
                var ctx = canvas.getContext("2d");
                ctx.drawImage(image, 0, 0, canvas.width, canvas.height);
                var data = {};
                data['base64'] = canvas.toDataURL();
                data['filename'] = filename.replace(/\//g, '_');
                deferred.resolve(data);
            };
            return deferred.promise()
        }
    },
    packageImages: function (folderName, imgInfos) {
        if (!imgInfos.length) {
            return;
        }

        var img = batchDwonloadImagesLib.zip.folder(folderName);
        for (var i = 0; i < imgInfos.length; i++) {
            if(imgInfos[i].name.substring(imgInfos[i].name.length - 5) == '.avif') {
              imgInfos[i].name = imgInfos[i].name.substring(0, imgInfos[i].name.length - 5);
            }

            batchDwonloadImagesLib.getBase64(imgInfos[i].url, imgInfos[i].name).then(function (data) {
                img.file(data['filename'], data['base64'].substring(22), {base64: true});
                batchDwonloadImagesLib.downloadedImages++;
                if (batchDwonloadImagesLib.totalImages == batchDwonloadImagesLib.downloadedImages && batchDwonloadImagesLib.totalImages != 0){
                    layui.layer.closeAll();
                    showSimpleMessage('图片打包压缩中...', -1);
                    batchDwonloadImagesLib.zip.generateAsync({type: "blob"}).then(function (content) {
                        saveAs(content, batchDwonloadImagesLib.downloadFileName + '.zip');
                        layui.layer.closeAll();
                    });
                }
            }, function (err) {
                console.log(err);
            })
        }
    },
    // 图片打包下载
    download: function (filename, itemImages) {
        batchDwonloadImagesLib.downloadFileName = filename;
        batchDwonloadImagesLib.zip = new JSZip();
        batchDwonloadImagesLib.totalImages = 0;
        batchDwonloadImagesLib.downloadedImages = 0;

        for (let folderName in itemImages){
            batchDwonloadImagesLib.packageImages(folderName, itemImages[folderName]);
            batchDwonloadImagesLib.totalImages += itemImages[folderName].length;
            // console.log(batchDwonloadImagesLib.totalImages, itemImages[folderName].length);
        }
    },
};


// var itemImages = JSON.parse('{"主图":["https://img.alicdn.com/imgextra/i1/6000000005473/O1CN01S52e5H1qIgH8wAAND_!!6000000005473-0-tbvideo.jpg","https://gd4.alicdn.com/imgextra/i2/0/O1CN015RYWYP28WZMdENGdu_!!0-item_pic.jpg","https://gd3.alicdn.com/imgextra/i3/3032567940/O1CN01HxsNKc28WZMT2tBGj_!!3032567940.jpg","https://gd4.alicdn.com/imgextra/i4/3032567940/O1CN01fe8e3D28WZMPF1P5v_!!3032567940.jpg","https://gd1.alicdn.com/imgextra/i1/3032567940/O1CN01gYyZw928WZMPF1fi1_!!3032567940.jpg","https://gd3.alicdn.com/imgextra/i3/3032567940/O1CN01EGb9TE28WZMHJN9YO_!!3032567940.jpg"],"sku图":["https://gd1.alicdn.com/imgextra/i1/3032567940/O1CN01tU6ftz28WZMHJM0rm_!!3032567940.jpg","https://gd1.alicdn.com/imgextra/i3/3032567940/O1CN01yEzfOK28WZMRKYm4i_!!3032567940.jpg","https://gd3.alicdn.com/imgextra/i2/3032567940/O1CN01XjxX5328WZMRKYIyV_!!3032567940.jpg","https://gd3.alicdn.com/imgextra/i3/3032567940/O1CN01KZNzyP28WZMM1rMQ8_!!3032567940.jpg","https://gd1.alicdn.com/imgextra/i2/3032567940/O1CN01oZpYBl28WZMPpYvZp_!!3032567940.jpg","https://gd1.alicdn.com/imgextra/i2/3032567940/O1CN014bByw928WZMPpWFIK_!!3032567940.jpg","https://gd1.alicdn.com/imgextra/i2/3032567940/O1CN01hxjDYz28WZMSPXB4j_!!3032567940.jpg","https://gd1.alicdn.com/imgextra/i2/3032567940/O1CN01Ak1keE28WZMPpY7fu_!!3032567940.jpg","https://gd4.alicdn.com/imgextra/i3/3032567940/O1CN01kGcxvA28WZMPEzf2J_!!3032567940.jpg","https://gd2.alicdn.com/imgextra/i1/3032567940/O1CN01g1D04T28WZMMxtpau_!!3032567940.jpg","https://gd3.alicdn.com/imgextra/i4/3032567940/O1CN01Qc8skM28WZMPghpeZ_!!3032567940.jpg","https://gd1.alicdn.com/imgextra/i4/3032567940/O1CN01P1kRry28WZMTkn57P_!!3032567940.jpg","https://gd2.alicdn.com/imgextra/i2/3032567940/O1CN01LRRIyb28WZMPpWus1_!!3032567940.jpg","https://gd2.alicdn.com/imgextra/i1/3032567940/O1CN01M8Q6C028WZMRKbSRY_!!3032567940.jpg","https://gd3.alicdn.com/imgextra/i3/3032567940/O1CN01SWLZZp28WZMM1tNCR_!!3032567940.jpg"],"详情图":["https://img.alicdn.com/imgextra/i1/3032567940/O1CN01hZwCKG28WZMSPYJgF_!!3032567940.gif","https://img.alicdn.com/imgextra/i4/3032567940/O1CN010W5rWO28WZMMxrgV0_!!3032567940.jpg","https://img.alicdn.com/imgextra/i3/3032567940/O1CN017dH9IG28WZMQl6WGy_!!3032567940.jpg","https://img.alicdn.com/imgextra/i3/3032567940/O1CN01nRSbqa28WZMT2sVdL_!!3032567940.jpg","https://img.alicdn.com/imgextra/i4/3032567940/O1CN01LyDoXO28WZMMxt5mw_!!3032567940.jpg","https://img.alicdn.com/imgextra/i1/3032567940/O1CN01xWXb9B28WZMMxsYXL_!!3032567940.jpg","https://img.alicdn.com/imgextra/i4/3032567940/O1CN01sozA4D28WZMPpXqzr_!!3032567940.jpg","https://img.alicdn.com/imgextra/i4/3032567940/O1CN015uCUAL28WZMSPYid3_!!3032567940.jpg","https://img.alicdn.com/imgextra/i1/3032567940/O1CN01rwFU8F28WZMT2uO5U_!!3032567940.jpg","https://img.alicdn.com/imgextra/i1/3032567940/O1CN012iAQxO28WZMQl5qjx_!!3032567940.jpg","https://img.alicdn.com/imgextra/i1/3032567940/O1CN01MRYBIL28WZMPF0CDc_!!3032567940.jpg","https://img.alicdn.com/imgextra/i3/3032567940/O1CN01E7uoFa28WZMRKaJgu_!!3032567940.jpg","https://img.alicdn.com/imgextra/i3/3032567940/O1CN01bHhvSR28WZMM1uB1R_!!3032567940.jpg","https://img.alicdn.com/imgextra/i2/3032567940/O1CN01HFZvOB28WZMRKamoZ_!!3032567940.jpg","https://img.alicdn.com/imgextra/i3/3032567940/O1CN01UQTJO528WZMHJN1DY_!!3032567940.jpg","https://img.alicdn.com/imgextra/i4/3032567940/O1CN01Xz9uYK28WZMPpY3UL_!!3032567940.jpg","https://img.alicdn.com/imgextra/i1/3032567940/O1CN01o5pv8m28WZMM1sdO8_!!3032567940.jpg","https://img.alicdn.com/imgextra/i3/3032567940/O1CN01L9Q64D28WZMT2un2y_!!3032567940.jpg","https://img.alicdn.com/imgextra/i3/3032567940/O1CN011YU1Xv28WZMMxrHc2_!!3032567940.jpg","https://img.alicdn.com/imgextra/i3/3032567940/O1CN01FfYhzX28WZMMxqXqx_!!3032567940.jpg"]}');
// batchDwonloadImagesLib.download('test', itemImages);
